// Production WhatsApp Bot - Mas Angga
// Version: 1.0.0 - Ready for cPanel/VPS Deployment

// Production Error Handling
process.on('uncaughtException', (error) => {
    console.error('💥 Uncaught Exception:', error);
    console.error('Stack:', error.stack);
    // Don't exit in production, just log
});

process.on('unhandledRejection', (reason, promise) => {
    console.error('💥 Unhandled Rejection at:', promise, 'reason:', reason);
    // Don't exit in production, just log
});

// Graceful shutdown
process.on('SIGTERM', () => {
    console.log('💤 SIGTERM received, shutting down gracefully...');
    if (global.whatsappSocket) {
        global.whatsappSocket.end();
    }
    process.exit(0);
});

process.on('SIGINT', () => {
    console.log('💤 SIGINT received, shutting down gracefully...');
    if (global.whatsappSocket) {
        global.whatsappSocket.end();
    }
    process.exit(0);
});

// Crypto polyfill untuk Baileys
const crypto = require('crypto');
if (typeof globalThis.crypto === 'undefined') {
    globalThis.crypto = {
        getRandomValues: (arr) => {
            if (arr.constructor === Uint8Array) {
                return crypto.randomFillSync(arr);
            }
            return crypto.randomFillSync(new Uint8Array(arr));
        },
        subtle: crypto.webcrypto?.subtle || {}
    };
}

if (typeof global !== 'undefined' && !global.crypto) {
    global.crypto = globalThis.crypto;
}

// Main Dependencies
const { makeWASocket, DisconnectReason, useMultiFileAuthState } = require('@whiskeysockets/baileys');
const { Boom } = require('@hapi/boom');
const qrcode = require('qrcode-terminal');
const P = require('pino');
const AIService = require('./ai-service');
const TikTokService = require('./tiktok-service');
const EwalletService = require('./ewallet-service');
const HealthCheckServer = require('./health-server');
require('dotenv').config();

// Initialize health check server untuk production monitoring
const healthServer = new HealthCheckServer();
const PORT = process.env.PORT || 3000;
healthServer.start(PORT);

class WhatsAppBot {
    constructor() {
        // Initialize AI Service
        const groqKey = process.env.GROQ_API_KEY;
        const openaiKey = process.env.OPENAI_API_KEY;
        
        if (groqKey) {
            this.aiService = new AIService(groqKey, 'groq');
            console.log('🤖 Using Groq AI (Free)');
            healthServer.updateStatus('ai-ready-groq');
        } else if (openaiKey) {
            this.aiService = new AIService(openaiKey, 'openai');
            console.log('🤖 Using OpenAI');
            healthServer.updateStatus('ai-ready-openai');
        } else {
            this.aiService = new AIService(null, 'demo');
            console.log('⚠️ Demo mode - Setup API key untuk fitur AI lengkap');
            healthServer.updateStatus('demo-mode');
        }
        
        this.botName = process.env.BOT_NAME || 'Mas Angga';
        this.socket = null;
        this.tiktokService = new TikTokService();
        this.ewalletService = new EwalletService();
        this.userNames = new Map();
        
        // Set global reference for health checks
        global.whatsappBot = this;
        
        console.log(`🚀 ${this.botName} initializing...`);
    }

    async startBot() {
        try {
            healthServer.updateStatus('connecting');
            
            const { state, saveCreds } = await useMultiFileAuthState('./auth');
            
            // Silent logger untuk production
            const logger = P({ 
                level: 'silent',
                prettyPrint: false,
                timestamp: false 
            });
            
            this.socket = makeWASocket({
                auth: state,
                printQRInTerminal: false, // Disable terminal QR untuk production
                logger: logger,
                browser: ['Mas Angga Bot', 'Chrome', '1.0.0'],
                generateHighQualityLinkPreview: true,
                markOnlineOnConnect: true
            });

            // Set global reference
            global.whatsappSocket = this.socket;

            this.socket.ev.on('creds.update', saveCreds);
            this.socket.ev.on('connection.update', this.handleConnection.bind(this));
            this.socket.ev.on('messages.upsert', this.handleMessage.bind(this));

            console.log('🤖 WhatsApp Bot starting...');
            healthServer.updateStatus('starting');
            
        } catch (error) {
            console.error('❌ Error starting bot:', error);
            healthServer.updateStatus('error');
            
            // Auto-retry dengan delay
            setTimeout(() => {
                console.log('🔄 Retrying in 10 seconds...');
                this.startBot();
            }, 10000);
        }
    }

    handleConnection(update) {
        const { connection, lastDisconnect, qr } = update;
        
        if (qr) {
            console.log('📱 QR Code generated for WhatsApp authentication');
            console.log(`🌐 Scan QR via web: http://localhost:${PORT}/qr`);
            
            // Update QR di health server untuk web interface
            healthServer.updateQR(qr);
            healthServer.updateStatus('qr-ready');
            
            // Tampilkan QR di terminal juga (untuk development)
            if (process.env.NODE_ENV !== 'production') {
                qrcode.generate(qr, { small: true });
            }
        }
        
        if (connection === 'close') {
            const shouldReconnect = (lastDisconnect?.error instanceof Boom) ?
                lastDisconnect.error.output?.statusCode !== DisconnectReason.loggedOut : true;
                
            console.log('🔌 Connection closed:', lastDisconnect?.error?.message || 'Unknown error');
            healthServer.updateStatus('disconnected');
            
            if (shouldReconnect) {
                console.log('🔄 Reconnecting in 5 seconds...');
                setTimeout(() => {
                    this.startBot();
                }, 5000);
            } else {
                console.log('❌ Bot logged out');
                healthServer.updateStatus('logged-out');
            }
        } else if (connection === 'open') {
            console.log('✅ Bot connected to WhatsApp!');
            console.log(`🤖 ${this.botName} ready to serve!`);
            console.log(`🌐 Bot dashboard: http://localhost:${PORT}/`);
            
            healthServer.updateStatus('online');
            healthServer.updateQR(null); // Clear QR when connected
        } else if (connection === 'connecting') {
            console.log('🔄 Connecting to WhatsApp...');
            healthServer.updateStatus('connecting');
        }
    }

    async handleMessage(messageUpdate) {
        try {
            const message = messageUpdate.messages[0];
            if (!message.message || message.key.fromMe) return;

            const from = message.key.remoteJid;
            const text = this.getMessageText(message);
            const senderName = message.pushName || 'User';

            console.log(`📩 Message from ${senderName}: ${text.substring(0, 50)}...`);

            if (text) {
                await this.processMessage(from, text, senderName);
            }
        } catch (error) {
            console.error('❌ Error handling message:', error);
        }
    }

    getMessageText(message) {
        return message.message?.conversation ||
               message.message?.extendedTextMessage?.text ||
               message.message?.imageMessage?.caption ||
               '';
    }

    async processMessage(from, text, senderName) {
        try {
            const command = text.toLowerCase().trim();
            this.userNames.set(from, senderName);

            // TikTok URL detection (highest priority)
            if (this.tiktokService.shouldProcessMessage(text)) {
                await this.handleTikTokDownload(from, text, senderName);
                return;
            }

            // E-wallet check detection
            if (this.ewalletService.shouldProcessEwalletCheck(text)) {
                await this.handleEwalletCheck(from, text, senderName);
                return;
            }

            // Bot commands
            if (command.startsWith('/help')) {
                await this.sendHelpMessage(from, senderName);
            } else if (command === '/ping') {
                await this.sendMessage(from, '🏓 Pong! Bot online dan siap membantu.');
            } else if (command === '/clear') {
                this.aiService.clearHistory(from);
                await this.sendMessage(from, '🧹 History cleared! Fresh start!');
            } else if (command === '/status') {
                await this.sendStatusMessage(from, senderName);
            } else {
                // Natural chat processing
                await this.handleNaturalChat(from, text, senderName);
            }
        } catch (error) {
            console.error('❌ Error processing message:', error);
            await this.sendMessage(from, '❌ Maaf, ada error. Coba lagi ya!');
        }
    }

    async handleNaturalChat(from, message, senderName) {
        try {
            const result = await this.aiService.chatNaturally(message, from, senderName);
            await this.sendMessage(from, result.response);
            
            console.log(`🤖 Mode: ${result.mode}, Emotion: ${result.emotion}`);
        } catch (error) {
            console.error('❌ Error in natural chat:', error);
            await this.sendMessage(from, 'Maaf, ada gangguan teknis. Coba lagi ya!');
        }
    }

    async handleTikTokDownload(from, text, senderName) {
        try {
            const url = this.tiktokService.extractTikTokUrl(text);
            if (!url) {
                await this.sendMessage(from, `Maaf ${senderName}, URL TikTok tidak valid.`);
                return;
            }

            await this.sendMessage(from, `📥 Downloading TikTok video for ${senderName}...`);

            const result = await this.tiktokService.downloadVideo(url);
            
            if (result.success && result.data) {
                const videoInfo = this.tiktokService.formatVideoInfo(result.data);
                await this.sendMessage(from, videoInfo);

                if (result.data.videoUrl) {
                    try {
                        await this.socket.sendMessage(from, {
                            video: { url: result.data.videoUrl },
                            caption: `🎬 Video TikTok dari ${result.data.author}\n📝 ${result.data.title}`,
                            mimetype: 'video/mp4'
                        });
                        console.log(`✅ TikTok video sent to ${senderName}`);
                    } catch (sendError) {
                        console.error('❌ Error sending video:', sendError);
                        await this.sendMessage(from, `Video info retrieved! Visit: ${url}`);
                    }
                } else {
                    await this.sendMessage(from, `Info retrieved! Visit: ${url}`);
                }
            } else {
                await this.sendMessage(from, `❌ Failed to download TikTok video.`);
            }

        } catch (error) {
            console.error('❌ Error in TikTok download:', error);
            await this.sendMessage(from, `❌ Error downloading TikTok: ${error.message}`);
        }
    }

    async handleEwalletCheck(from, text, senderName) {
        try {
            console.log(`[E-wallet] ${senderName} requested: ${text}`);
            
            await this.sendMessage(from, `🔍 Checking account... Please wait ${senderName}...`);
            
            const result = await this.ewalletService.handleAccountCheck(text);
            await this.sendMessage(from, result);
            
            console.log(`✅ E-wallet check completed for ${senderName}`);
            
        } catch (error) {
            console.error('❌ Error in e-wallet check:', error);
            await this.sendMessage(from, `❌ Error checking account. Try again!`);
        }
    }

    async sendStatusMessage(from, senderName) {
        const uptime = Math.floor(process.uptime());
        const status = `📊 **Bot Status untuk ${senderName}**

🤖 Bot: ${this.botName}
⏰ Uptime: ${uptime} seconds
💻 Memory: ${Math.round(process.memoryUsage().heapUsed / 1024 / 1024)} MB
🔗 Connected: ${this.socket?.user?.id ? '✅ Yes' : '❌ No'}
🌐 Dashboard: http://localhost:${PORT}/

Type /help for commands!`;

        await this.sendMessage(from, status);
    }

    async sendHelpMessage(from, senderName) {
        const helpText = `🤖 **${this.botName} - Help Guide**

**Halo ${senderName}! Ini yang bisa aku bantu:**

💝 **Natural Chat**
Langsung chat aja! Aku bisa:
• Jadi teman curhat
• Bantu pelajaran
• Ngobrol santai

🎬 **TikTok Download**
Kirim URL TikTok, aku download tanpa watermark!

💳 **E-wallet Check**
Format: "cek dana 081234567890"
Supported: DANA, OVO, GoPay, ShopeePay, LinkAja, Bank

**Commands:**
• /help - Bantuan
• /ping - Test connection
• /clear - Clear chat history
• /status - Bot status

**Tips:**
✅ Chat natural, no commands needed!
✅ Kirim URL TikTok langsung
✅ Bot detect emotion & subjects automatically

Selamat menggunakan! 🚀`;

        await this.sendMessage(from, helpText);
    }

    async sendMessage(jid, text) {
        try {
            await this.socket.sendMessage(jid, { text });
        } catch (error) {
            console.error('❌ Error sending message:', error);
        }
    }
}

// Start bot if this file is run directly
if (require.main === module) {
    console.log('🚀 Starting Mas Angga WhatsApp Bot...');
    console.log('📱 Production mode ready for deployment');
    
    const bot = new WhatsAppBot();
    bot.startBot().catch(console.error);
    
    // Memory monitoring for production
    setInterval(() => {
        const used = process.memoryUsage();
        if (used.heapUsed / 1024 / 1024 > 200) { // Alert if memory > 200MB
            console.log('⚠️ High memory usage:', Math.round(used.heapUsed / 1024 / 1024), 'MB');
        }
    }, 60000); // Check every minute
}

module.exports = WhatsAppBot;
