// Setup Configuration Server - Standalone untuk konfigurasi awal
const express = require('express');
const fs = require('fs');
const path = require('path');

class ConfigServer {
    constructor() {
        this.app = express();
        this.setupMiddleware();
        this.setupRoutes();
    }

    setupMiddleware() {
        this.app.use(express.json());
        this.app.use(express.urlencoded({ extended: true }));
        this.app.use(express.static('.'));
    }

    setupRoutes() {
        // Main configuration page
        this.app.get('/', (req, res) => {
            res.sendFile(path.join(__dirname, 'config.html'));
        });

        // Check if .env exists
        this.app.get('/api/check-setup', (req, res) => {
            const envExists = fs.existsSync('.env');
            const hasGroqKey = process.env.GROQ_API_KEY && process.env.GROQ_API_KEY.length > 10;
            
            res.json({
                env_exists: envExists,
                groq_configured: hasGroqKey,
                setup_complete: envExists && hasGroqKey
            });
        });

        // Get current configuration
        this.app.get('/api/config', (req, res) => {
            try {
                // Load environment variables dari .env jika ada
                if (fs.existsSync('.env')) {
                    const envContent = fs.readFileSync('.env', 'utf8');
                    const envVars = {};
                    
                    envContent.split('\n').forEach(line => {
                        const [key, value] = line.split('=');
                        if (key && value) {
                            envVars[key.trim()] = value.trim();
                        }
                    });
                    
                    res.json({
                        bot_name: envVars.BOT_NAME || 'Mas Angga',
                        port: envVars.PORT || '3000',
                        memory_limit: envVars.MEMORY_LIMIT || '512',
                        groq_api_key: envVars.GROQ_API_KEY ? envVars.GROQ_API_KEY.slice(-8) : '',
                        node_env: envVars.NODE_ENV || 'production'
                    });
                } else {
                    // Default config
                    res.json({
                        bot_name: 'Mas Angga',
                        port: '3000',
                        memory_limit: '512',
                        groq_api_key: '',
                        node_env: 'production'
                    });
                }
            } catch (error) {
                res.status(500).json({ error: error.message });
            }
        });

        // Save configuration
        this.app.post('/api/config', (req, res) => {
            try {
                const config = req.body;
                
                // Validate Groq API key
                if (!config.groq_api_key || config.groq_api_key.length < 10) {
                    return res.status(400).json({ 
                        error: 'Groq API key harus diisi dan valid (minimal 10 karakter)' 
                    });
                }

                // Create .env content
                let envContent = '';
                envContent += `NODE_ENV=production\n`;
                envContent += `BOT_NAME=${config.bot_name || 'Mas Angga'}\n`;
                envContent += `PORT=${config.port || '3000'}\n`;
                envContent += `GROQ_API_KEY=${config.groq_api_key}\n`;
                envContent += `MAX_RETRIES=3\n`;
                envContent += `REQUEST_TIMEOUT=30000\n`;
                envContent += `MEMORY_LIMIT=${config.memory_limit || '512'}\n`;

                // Write .env file
                fs.writeFileSync('.env', envContent);

                res.json({ 
                    message: '✅ Konfigurasi berhasil disimpan! Anda bisa mulai bot sekarang.',
                    success: true,
                    setup_complete: true
                });

            } catch (error) {
                res.status(500).json({ error: error.message });
            }
        });

        // Bot control (simplified untuk setup)
        this.app.post('/api/bot/start', (req, res) => {
            try {
                // Check if .env exists
                if (!fs.existsSync('.env')) {
                    return res.status(400).json({ 
                        error: 'Konfigurasi belum tersimpan. Silakan isi form terlebih dahulu.' 
                    });
                }

                const { spawn } = require('child_process');
                const botProcess = spawn('node', ['index-production.js'], {
                    stdio: 'pipe',
                    cwd: __dirname,
                    detached: true
                });

                botProcess.unref();

                res.json({ 
                    message: '🚀 Bot sedang dimulai... Silakan buka /qr untuk scan QR code.',
                    success: true
                });

            } catch (error) {
                res.status(500).json({ error: error.message });
            }
        });

        // Status check
        this.app.get('/api/status', (req, res) => {
            const envExists = fs.existsSync('.env');
            const authExists = fs.existsSync('auth') && fs.readdirSync('auth').length > 0;
            
            res.json({
                config_ready: envExists,
                auth_ready: authExists,
                setup_status: envExists ? 'Konfigurasi OK' : 'Perlu setup',
                connection_status: authExists ? 'Tersimpan' : 'Belum connect',
                memory_usage_mb: Math.round(process.memoryUsage().heapUsed / 1024 / 1024),
                node_version: process.version
            });
        });

        // Redirect QR to bot server
        this.app.get('/qr', (req, res) => {
            const botPort = process.env.PORT || '3000';
            res.redirect(`http://localhost:${botPort}/qr`);
        });

        // Simple landing page
        this.app.get('/dashboard', (req, res) => {
            res.send(`
                <!DOCTYPE html>
                <html>
                <head>
                    <title>Mas Angga Bot Setup</title>
                    <meta charset="UTF-8">
                    <style>
                        body { font-family: Arial, sans-serif; max-width: 600px; margin: 50px auto; padding: 20px; }
                        .status { padding: 10px; border-radius: 5px; margin: 10px 0; }
                        .success { background: #d4edda; color: #155724; }
                        .warning { background: #fff3cd; color: #856404; }
                        .btn { background: #007bff; color: white; padding: 10px 20px; text-decoration: none; border-radius: 5px; }
                    </style>
                </head>
                <body>
                    <h1>🤖 Mas Angga Bot - Setup Dashboard</h1>
                    <div class="status success">✅ Configuration server is running</div>
                    <p>Akses halaman konfigurasi untuk setup bot:</p>
                    <a href="/" class="btn">⚙️ Buka Konfigurasi</a>
                    <a href="/qr" class="btn">📱 Lihat QR Code</a>
                </body>
                </html>
            `);
        });
    }

    start(port = 8080) {
        this.app.listen(port, () => {
            console.log('');
            console.log('🚀 ============================================');
            console.log('   MAS ANGGA BOT - CONFIGURATION SERVER');
            console.log('🚀 ============================================');
            console.log('');
            console.log(`🌐 Buka browser dan akses:`);
            console.log(`   http://localhost:${port}`);
            console.log('');
            console.log(`📋 Setup Steps:`);
            console.log(`   1. Isi Groq API Key di form`);
            console.log(`   2. Klik 'Simpan Konfigurasi'`);
            console.log(`   3. Klik 'Start Bot'`);
            console.log(`   4. Scan QR Code di WhatsApp`);
            console.log('');
            console.log(`🔑 Get Groq API Key (FREE):`);
            console.log(`   https://console.groq.com/keys`);
            console.log('');
            console.log('🚀 ============================================');
        });
    }
}

// Jika dijalankan langsung
if (require.main === module) {
    const server = new ConfigServer();
    const port = process.argv[2] || 8080;
    server.start(port);
}

module.exports = ConfigServer;
