const OpenAI = require('openai');
const Groq = require('groq-sdk');

class AIService {
    constructor(apiKey, provider = 'groq') {
        this.provider = provider.toLowerCase();
        
        if (this.provider === 'openai' && apiKey) {
            this.openai = new OpenAI({
                apiKey: apiKey
            });
        } else if (this.provider === 'groq' && apiKey) {
            this.groq = new Groq({
                apiKey: apiKey
            });
        } else {
            // Fallback ke mode demo jika tidak ada API key
            console.log('⚠️ Tidak ada API key valid, menggunakan mode demo');
            this.provider = 'demo';
        }
        
        this.conversationHistory = new Map();
        this.userNames = new Map(); // Track nama user yang sudah dikenali
        this.firstTimeUsers = new Set(); // Track user yang pertama kali chat
    }
      // Chat natural tanpa command - otomatis detect mode
    async chatNaturally(message, from, userName = 'User') {
        try {
            // Cek apakah ini user pertama kali
            const isFirstTime = !this.conversationHistory.has(from);
            
            // Simpan riwayat percakapan
            if (!this.conversationHistory.has(from)) {
                this.conversationHistory.set(from, []);
                this.firstTimeUsers.add(from);
            }
            
            const history = this.conversationHistory.get(from);
              // Cek apakah user belum kasih nama yang proper
            const hasProperName = this.userNames.has(from) && 
                                 this.userNames.get(from) !== 'User' && 
                                 this.userNames.get(from) !== 'Unknown';
            
            // Jika first time atau belum ada nama proper, minta perkenalan
            if (isFirstTime || !hasProperName) {
                return await this.handleFirstTimeIntroduction(from, message, userName);
            }
            
            // Gunakan nama yang sudah disimpan
            const savedName = this.userNames.get(from) || userName;
            
            history.push({ role: 'user', content: message, name: savedName });
            
            // Keep only last 10 messages untuk efisiensi
            if (history.length > 10) {
                history.splice(0, history.length - 10);
            }
            
            // Deteksi otomatis apakah ini curhat atau belajar
            const mode = this.detectMode(message);
            
            // Buat context berdasarkan mode dengan nama yang benar
            const systemPrompt = this.getSystemPrompt(mode, savedName);
            
            // Generate response
            const response = await this.generateResponse(systemPrompt, history);
            
            // Simpan response ke history
            history.push({ role: 'assistant', content: response });
            
            return {
                response: response,
                mode: mode,
                emotion: this.detectEmotion(message),
                subject: mode === 'study' ? this.detectSubject(message) : null
            };
            
        } catch (error) {
            console.error('❌ Error in chatNaturally:', error);
            return this.getFallbackResponse(message);
        }
    }
    
    detectMode(message) {
        const studyKeywords = [
            'belajar', 'pelajaran', 'tugas', 'pr', 'ujian', 'soal', 'rumus', 'jelaskan',
            'matematika', 'fisika', 'kimia', 'biologi', 'sejarah', 'geografi', 'bahasa',
            'english', 'grammar', 'essay', 'hitungan', 'cara menghitung', 'bagaimana',
            'kenapa', 'mengapa', 'apa itu', 'definisi', 'pengertian', 'contoh'
        ];
        
        const curhatKeywords = [
            'curhat', 'sedih', 'senang', 'marah', 'kesal', 'stress', 'galau', 'bingung',
            'takut', 'cemas', 'bahagia', 'gembira', 'kecewa', 'lelah', 'cape',
            'halo', 'hai', 'hi', 'gimana', 'apa kabar', 'lagi apa', 'chat'
        ];
        
        const lowerMessage = message.toLowerCase();
        
        // Check study keywords first (more specific)
        if (studyKeywords.some(keyword => lowerMessage.includes(keyword))) {
            return 'study';
        }
        
        // Check curhat keywords or default to curhat for general chat
        if (curhatKeywords.some(keyword => lowerMessage.includes(keyword)) || 
            lowerMessage.length < 50) { // Short messages usually chat
            return 'curhat';
        }
        
        // Default to curhat for natural conversation
        return 'curhat';
    }    getSystemPrompt(mode, userName) {
        if (mode === 'study') {            return `Kamu adalah Mas Angga, teman yang pintar dan suka bantu orang belajar. 
Kamu lagi chat sama ${userName}. Tugasmu bantu ${userName} dengan pelajaran atau tugas. Jelasin dengan cara yang gampang dimengerti, 
kasih contoh yang simpel, dan step-by-step yang jelas. Ngobrol aja santai kayak teman biasa, jangan formal atau kayak robot. 
ATURAN PENTING: 
1. SELALU panggil dia dengan nama "${userName}", JANGAN PERNAH pakai "kamu" 
2. Ganti semua "kamu" dengan "${userName}" dalam responsmu
3. Contoh SALAH: "gimana kabar kamu?" - Contoh BENAR: "gimana kabar ${userName}?"
4. Contoh SALAH: "apa yang kamu mau?" - Contoh BENAR: "apa yang ${userName} mau?"
Respon seperti human pada umumnya tanpa embel-embel "sebagai AI" atau semacamnya.`;
        } else {            return `Kamu adalah Mas Angga, teman yang baik, pengertian, dan suka dengerin orang curhat.
Kamu lagi chat sama ${userName}. Kamu jago dengerin dan kasih dukungan ke ${userName}. Ngobrol natural kayak teman dekat yang peduli banget. 
Pakai bahasa Indonesia yang hangat, santai, dan relate. Bisa ngobrol tentang apa aja - perasaan, kehidupan, atau hal random. 
ATURAN PENTING: 
1. SELALU panggil dia dengan nama "${userName}", JANGAN PERNAH pakai "kamu" 
2. Ganti semua "kamu" dengan "${userName}" dalam responsmu
3. Contoh SALAH: "gimana kabar kamu?" - Contoh BENAR: "gimana kabar ${userName}?"
4. Contoh SALAH: "aku ngerti perasaan kamu" - Contoh BENAR: "aku ngerti perasaan ${userName}"
Respon seperti human pada umumnya tanpa embel-embel "sebagai AI" atau semacamnya. Just be a good friend!`;
        }
    }
    
    async generateResponse(systemPrompt, history) {
        if (this.provider === 'groq' && this.groq) {
            return await this.generateGroqResponse(systemPrompt, history);
        } else if (this.provider === 'openai' && this.openai) {
            return await this.generateOpenAIResponse(systemPrompt, history);
        } else {
            return this.getDemoResponse(history[history.length - 1]?.content || '');
        }
    }
    
    async generateGroqResponse(systemPrompt, history) {
        try {
            const messages = [
                { role: 'system', content: systemPrompt },
                ...history.slice(-8) // Last 8 messages for context
            ];
              const completion = await this.groq.chat.completions.create({
                messages: messages,
                model: 'llama-3.1-8b-instant', // Updated to current Groq model
                temperature: 0.7,
                max_tokens: 500,
                top_p: 0.9
            });
            
            return completion.choices[0]?.message?.content || 'Maaf, aku lagi bingung nih 😅';
            
        } catch (error) {
            console.error('❌ Groq API Error:', error);
            throw error;
        }
    }
    
    async generateOpenAIResponse(systemPrompt, history) {
        try {
            const messages = [
                { role: 'system', content: systemPrompt },
                ...history.slice(-8)
            ];
            
            const completion = await this.openai.chat.completions.create({
                model: 'gpt-3.5-turbo',
                messages: messages,
                temperature: 0.7,
                max_tokens: 500
            });
            
            return completion.choices[0]?.message?.content || 'Maaf, aku lagi error nih 😅';
            
        } catch (error) {
            console.error('❌ OpenAI API Error:', error);
            throw error;
        }
    }
      getDemoResponse(message) {
        const responses = [
            "Halo! Gimana kabarnya? 😊",
            "Wah menarik banget yang kamu bilang! Cerita lebih lanjut dong",
            "Aku ngerti banget perasaan kamu. Mau cerita lebih detail?",
            "Hmm, itu pertanyaan bagus nih. Gimana kalau kita bahas bareng-bareng?",
            "Oh gitu ya? Aku sih setuju sama kamu",
            "Waduh, pasti seru tuh! Cerita dong gimana rasanya",
            "Iya nih, aku juga pernah ngalamin hal yang mirip"
        ];
        
        return responses[Math.floor(Math.random() * responses.length)];
    }
    
    detectEmotion(message) {
        const emotions = {
            sedih: ['sedih', 'kecewa', 'galau', 'down', 'stress', 'lelah', 'cape', 'putus asa'],
            marah: ['marah', 'kesal', 'dongkol', 'bete', 'sebel', 'jengkel'],
            senang: ['senang', 'bahagia', 'gembira', 'excited', 'happy', 'seru'],
            takut: ['takut', 'cemas', 'khawatir', 'nervous', 'deg-degan'],
            bingung: ['bingung', 'pusing', 'stuck', 'gatau', 'gimana']
        };
        
        const lowerMessage = message.toLowerCase();
        
        for (const [emotion, keywords] of Object.entries(emotions)) {
            if (keywords.some(keyword => lowerMessage.includes(keyword))) {
                return emotion;
            }
        }
        
        return 'netral';
    }
    
    detectSubject(message) {
        const subjects = {
            matematika: ['math', 'matematika', 'hitung', 'rumus', 'aljabar', 'geometri', 'kalkulus'],
            fisika: ['fisika', 'gaya', 'energi', 'listrik', 'magnet', 'gelombang', 'cahaya'],
            kimia: ['kimia', 'reaksi', 'molekul', 'atom', 'senyawa', 'unsur'],
            biologi: ['biologi', 'sel', 'dna', 'tumbuhan', 'hewan', 'evolusi', 'genetika'],
            bahasa: ['grammar', 'tata bahasa', 'essay', 'menulis', 'english', 'indonesia'],
            sejarah: ['sejarah', 'perang', 'kerajaan', 'kolonial', 'kemerdekaan'],
            geografi: ['geografi', 'peta', 'benua', 'negara', 'iklim', 'bumi']
        };
        
        const lowerMessage = message.toLowerCase();
        
        for (const [subject, keywords] of Object.entries(subjects)) {
            if (keywords.some(keyword => lowerMessage.includes(keyword))) {
                return subject;
            }
        }
        
        return 'umum';
    }
      getFallbackResponse(message) {
        const fallbacks = [
            "Eh maaf, koneksi aku lagi bermasalah nih. Coba lagi ya!",
            "Waduh, ada gangguan teknis. Tapi aku masih di sini kok!",
            "Oops, ada masalah sebentar. Mau coba chat lagi?",
            "Maaf ya, lagi ada kendala teknis. Coba lagi dong!"
        ];
        
        return {
            response: fallbacks[Math.floor(Math.random() * fallbacks.length)],
            mode: 'curhat',
            emotion: 'netral',
            subject: null
        };
    }
      clearHistory(from) {
        this.conversationHistory.delete(from);
        this.userNames.delete(from);
        this.firstTimeUsers.add(from); // Reset ke first time user
        return true;
    }
    
    async handleFirstTimeIntroduction(from, message, userName) {
        try {
            const history = this.conversationHistory.get(from) || [];
            
            // Cek apakah user sudah memberikan nama di pesan ini
            const namePattern = /nama\s+(?:saya|aku|gue|gw)\s+([a-zA-Z]+)|(?:saya|aku|gue|gw)\s+([a-zA-Z]+)(?:\s|$)/i;
            const nameMatch = message.match(namePattern);
            
            if (nameMatch) {
                // Ekstrak nama dari pesan
                const extractedName = nameMatch[1] || nameMatch[2];
                this.userNames.set(from, extractedName);
                this.firstTimeUsers.delete(from);
                  // Response setelah tau nama
                const welcomeResponses = [
                    `Senang kenal, ${extractedName}! Aku Mas Angga. Aku di sini kalau ${extractedName} mau curhat atau butuh bantuan belajar. Gimana hari ini?`,
                    `Hai ${extractedName}! Aku Mas Angga, teman chatmu. Kalau ${extractedName} lagi stress, sedih, atau mau belajar, aku siap dengarin kok!`,
                    `Nice to meet you, ${extractedName}! Aku Mas Angga. Mau cerita-cerita atau ada yang mau ditanyain soal pelajaran?`
                ];
                
                const response = welcomeResponses[Math.floor(Math.random() * welcomeResponses.length)];
                
                // Simpan ke history
                history.push({ role: 'user', content: message, name: extractedName });
                history.push({ role: 'assistant', content: response });
                
                return {
                    response: response,
                    mode: 'curhat',
                    emotion: 'senang',
                    subject: null
                };
                  } else {
                // Perkenalan pertama kali - minta nama
                const introResponses = [
                    "Hai! Aku Mas Angga 😊 Senang ketemu sama kamu! Siapa nama kamu? Aku suka ngobrol dan bantuin orang-orang kok.",
                    "Halo! Aku Mas Angga, nice to meet you! Nama kamu siapa? Aku di sini kalau mau curhat atau butuh bantuan belajar.",
                    "Hi there! Aku Mas Angga 🙋‍♂️ Boleh kenalan? Nama kamu apa? Aku suka dengarin cerita dan bantuin dengan pelajaran."
                ];
                
                const response = introResponses[Math.floor(Math.random() * introResponses.length)];
                
                // Simpan ke history
                history.push({ role: 'user', content: message, name: 'Unknown' });
                history.push({ role: 'assistant', content: response });
                
                return {
                    response: response,
                    mode: 'curhat',
                    emotion: 'netral',
                    subject: null
                };
            }
              } catch (error) {
            console.error('❌ Error in handleFirstTimeIntroduction:', error);
            return {
                response: "Hai! Aku Mas Angga 😊 Senang ketemu kamu! Nama kamu siapa?",
                mode: 'curhat',
                emotion: 'netral',
                subject: null
            };
        }
    }
}

module.exports = AIService;
