# 🏠 Deploy Bot WhatsApp ke Hosting Pribadi

## 🎯 Panduan Lengkap untuk Hosting & Domain Sendiri

### 📋 Checklist Persyaratan Hosting

**✅ Yang Harus Ada:**
- Node.js support (minimal v16+)
- SSH access atau terminal
- Port 3000+ bisa diakses
- Memory minimal 512MB
- Disk space 1GB+
- Bandwidth unlimited/tinggi

**❓ Cek Hosting Anda:**
```bash
# Login SSH dan cek:
node --version    # Harus v16+
npm --version     # Harus ada
pm2 --version     # Install jika belum ada
```

---

## 🚀 METHOD 1: cPanel dengan Node.js (Paling Umum)

### Step 1: Persiapan File Upload

**File yang dibutuhkan:**
```
whatsapp-bot-mas-angga/
├── index-production.js     # Main bot file
├── ai-service.js           # AI service
├── ewallet-service.js      # E-wallet checker
├── tiktok-service.js       # TikTok downloader
├── health-server.js        # Health monitoring
├── package.json            # Dependencies
├── .env                    # Environment variables
├── ecosystem.config.js     # PM2 config
└── README.md               # Documentation
```

### Step 2: Setup Environment Variables

**Buat file .env:**
```env
# Bot Configuration
NODE_ENV=production
BOT_NAME=Mas Angga
PORT=3000

# API Keys
GROQ_API_KEY=your_groq_api_key_here

# Server Configuration
MAX_RETRIES=3
REQUEST_TIMEOUT=30000
MEMORY_LIMIT=512
```

### Step 3: Upload ke Hosting

**Via cPanel File Manager:**
1. Login ke cPanel
2. Buka **File Manager**
3. Masuk ke folder `public_html` atau `domains/yourdomain.com/public_html`
4. Buat folder baru: `whatsapp-bot`
5. Upload semua file bot ke folder ini
6. Extract file zip (jika upload dalam bentuk zip)

**Via FTP/SFTP:**
```bash
# Gunakan FileZilla atau WinSCP
Host: your-domain.com
Port: 21 (FTP) atau 22 (SFTP)
Username: your-cpanel-username
Password: your-cpanel-password

# Upload ke: /public_html/whatsapp-bot/
```

### Step 4: Install Dependencies

**Via SSH Terminal:**
```bash
# Login SSH
ssh username@your-domain.com

# Masuk ke folder bot
cd public_html/whatsapp-bot

# Install dependencies
npm install

# Atau jika error, gunakan:
npm install --production --no-optional
```

**Via cPanel Node.js App:**
1. Buka **Node.js Apps** di cPanel
2. Klik **Create Application**
3. Setup:
   - **Node.js Version**: 18.x
   - **Application Mode**: Production
   - **Application Root**: whatsapp-bot
   - **Application URL**: yourdomain.com/bot
   - **Application Startup File**: index-production.js

### Step 5: Setup Domain/Subdomain

**Option A: Subdomain (Recommended)**
```
bot.yourdomain.com → /public_html/whatsapp-bot
```

**Setup di cPanel:**
1. **Subdomains** → Create subdomain `bot`
2. **Document Root**: `/public_html/whatsapp-bot`
3. Bot akan bisa diakses di: `https://bot.yourdomain.com`

**Option B: Directory di Domain Utama**
```
yourdomain.com/bot → /public_html/whatsapp-bot
```

### Step 6: Start Bot

**Via PM2 (Recommended):**
```bash
# Install PM2 globally
npm install -g pm2

# Start bot dengan PM2
pm2 start ecosystem.config.js

# Save configuration
pm2 save

# Setup auto-restart
pm2 startup
```

**Via cPanel Node.js App:**
1. Buka **Node.js Apps**
2. Pilih aplikasi yang sudah dibuat
3. Klik **Start** atau **Restart**

---

## 🚀 METHOD 2: VPS/Dedicated Server

### Step 1: Server Setup

```bash
# Update server
sudo apt update && sudo apt upgrade -y

# Install Node.js 18
curl -fsSL https://deb.nodesource.com/setup_18.x | sudo -E bash -
sudo apt-get install -y nodejs

# Install PM2
npm install -g pm2

# Install Nginx (for domain)
sudo apt install nginx
```

### Step 2: Upload Bot Files

```bash
# Via Git (Recommended)
git clone https://github.com/yourusername/whatsapp-bot.git
cd whatsapp-bot

# Atau upload manual via SCP
scp -r ./whatsapp-bot root@your-server-ip:/var/www/
```

### Step 3: Configure Domain

**Nginx Configuration:**
```nginx
# /etc/nginx/sites-available/bot.yourdomain.com
server {
    listen 80;
    server_name bot.yourdomain.com;

    location / {
        proxy_pass http://localhost:3000;
        proxy_http_version 1.1;
        proxy_set_header Upgrade $http_upgrade;
        proxy_set_header Connection 'upgrade';
        proxy_set_header Host $host;
        proxy_set_header X-Real-IP $remote_addr;
        proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto $scheme;
        proxy_cache_bypass $http_upgrade;
    }
}
```

**Enable Site:**
```bash
sudo ln -s /etc/nginx/sites-available/bot.yourdomain.com /etc/nginx/sites-enabled/
sudo nginx -t
sudo systemctl reload nginx
```

### Step 4: SSL Certificate

```bash
# Install Certbot
sudo apt install certbot python3-certbot-nginx

# Get SSL certificate
sudo certbot --nginx -d bot.yourdomain.com

# Auto-renewal
sudo crontab -e
# Add: 0 12 * * * /usr/bin/certbot renew --quiet
```

---

## 🎯 METHOD 3: Shared Hosting (Workaround)

### Jika Hosting Tidak Support Node.js

**Option A: Webhook ke External Service**
```javascript
// webhook-forwarder.php
<?php
$webhook_url = "https://your-railway-app.herokuapp.com/webhook";
$data = file_get_contents('php://input');

$ch = curl_init($webhook_url);
curl_setopt($ch, CURLOPT_POST, 1);
curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
curl_setopt($ch, CURLOPT_HTTPHEADER, [
    'Content-Type: application/json'
]);
curl_exec($ch);
curl_close($ch);
?>
```

**Option B: Subdomain ke External**
```
# DNS CNAME Record
bot.yourdomain.com → your-railway-app.up.railway.app
```

---

## 🔧 Configuration Files untuk Hosting

### package.json (Production Ready)

```json
{
  "name": "mas-angga-whatsapp-bot",
  "version": "1.0.0",
  "description": "WhatsApp Bot AI untuk curhat, belajar, TikTok, dan e-wallet",
  "main": "index-production.js",
  "scripts": {
    "start": "node index-production.js",
    "dev": "node index.js",
    "pm2:start": "pm2 start ecosystem.config.js",
    "pm2:stop": "pm2 stop mas-angga-bot",
    "pm2:restart": "pm2 restart mas-angga-bot",
    "pm2:logs": "pm2 logs mas-angga-bot"
  },
  "engines": {
    "node": ">=16.0.0",
    "npm": ">=8.0.0"
  },
  "dependencies": {
    "@whiskeysockets/baileys": "^6.6.0",
    "groq-sdk": "^0.3.0",
    "axios": "^1.6.0",
    "qrcode-terminal": "^0.12.0",
    "qrcode": "^1.5.3",
    "express": "^4.18.0",
    "dotenv": "^16.0.0",
    "winston": "^3.8.0"
  }
}
```

### .htaccess (untuk cPanel)

```apache
# Redirect subdomain to Node.js app
RewriteEngine On
RewriteCond %{HTTP_HOST} ^bot\.yourdomain\.com$ [NC]
RewriteRule ^(.*)$ http://localhost:3000/$1 [P,L]

# Security headers
Header always set X-Content-Type-Options nosniff
Header always set X-Frame-Options DENY
Header always set X-XSS-Protection "1; mode=block"
```

---

## 📱 Akses Bot Setelah Deploy

### Health Check Interface
```
https://bot.yourdomain.com/health
```

### QR Code Scanner
```
https://bot.yourdomain.com/qr
```

### Bot Dashboard
```
https://bot.yourdomain.com/
```

### API Endpoint (Opsional)
```
https://bot.yourdomain.com/api/send
```

---

## 🔍 Troubleshooting Hosting

### 1. Bot Tidak Start

**Check Node.js:**
```bash
node --version  # Harus >= 16
npm --version   # Harus ada
```

**Check Dependencies:**
```bash
npm install --production
npm audit fix
```

**Check Logs:**
```bash
pm2 logs mas-angga-bot
# atau
cat ~/.pm2/logs/mas-angga-bot-error.log
```

### 2. Port Issues

**cPanel/Shared Hosting:**
- Gunakan port yang dialokasikan oleh hosting
- Biasanya 3000, 8080, atau yang ditentukan

**VPS:**
```bash
# Check port availability
netstat -tulpn | grep :3000
sudo ufw allow 3000
```

### 3. Memory Issues

**Optimize untuk Shared Hosting:**
```javascript
// Di index-production.js
process.env.NODE_OPTIONS = '--max-old-space-size=512';

// Cleanup memory periodically
setInterval(() => {
    if (global.gc) {
        global.gc();
    }
}, 30000);
```

### 4. QR Code Issues

**Via Web Interface:**
```
https://bot.yourdomain.com/qr
```

**Via SSH:**
```bash
pm2 logs mas-angga-bot | grep QR
```

---

## 💡 Best Practices untuk Hosting Pribadi

### 1. Security

```bash
# Change default SSH port
sudo nano /etc/ssh/sshd_config
# Port 2222

# Setup firewall
sudo ufw enable
sudo ufw allow 2222
sudo ufw allow 80
sudo ufw allow 443
```

### 2. Monitoring

```bash
# Setup monitoring
npm install -g pm2-logrotate
pm2 install pm2-server-monit

# Weekly restart
crontab -e
# 0 3 * * 0 pm2 restart mas-angga-bot
```

### 3. Backup

```bash
# Backup script
#!/bin/bash
DATE=$(date +%Y%m%d)
tar -czf "bot-backup-$DATE.tar.gz" auth/ .env
scp "bot-backup-$DATE.tar.gz" user@backup-server:/backups/
```

---

## 🎯 Recommended Setup

### Untuk Domain Pribadi:
```
Domain: yourdomain.com
Subdomain: bot.yourdomain.com
SSL: Let's Encrypt (gratis)
CDN: Cloudflare (gratis)
```

### File Structure:
```
/public_html/
├── yourdomain.com/          # Website utama
└── whatsapp-bot/            # Bot WhatsApp
    ├── index-production.js
    ├── auth/                # Session data
    ├── logs/                # Log files
    └── uploads/             # TikTok downloads
```

**Siap untuk setup?** Kasih tau jenis hosting Anda biar saya bisa guide lebih spesifik! 🚀
